<?PHP
if (! defined ( 'BASEPATH' )) exit ( 'No direct script access allowed' );

/**
 * @package direct-as-a-service
 * @subpackage models
 * @filesource
 */

/** */
require_once 'entity.php';

/**
 * @package direct-as-a-service
 * @subpackage models           
 */
class Disclosure extends Entity {
	static $database_group = 'mail_db';
	static $table = 'accounting_disclosure';
		
	protected static $_relationships = array( 'message' => array('type' => 'belongs_to', 'related_foreign_key' => 'messages_table_id') );	
	
	protected $_readonly_fields = array( 'hash', 'message_id', 'received_from', 'username');
	
	protected $_property_validation_rules = array('recipient' => 'string_like_an_email_address');
	
	public function __construct($values = array()){
		//make sure the message id is the first value set, because we'll need to use it to verify the hash value
		$message_foreign_key = static::related_foreign_key('message');
		if(array_key_exists($message_foreign_key, $values)){
			$values = array_merge(array($message_foreign_key => $values[$message_foreign_key]), $values);
		}		
		return parent::__construct($values);
	}
	
	
	public function has_message(){
		$message_foreign_key = static::related_foreign_key('message');
		return isset($message_foreign_key) && Message::formatted_like_an_id($this->$message_foreign_key);
	}
	
	
	/////////////////////////////
	// GETTERS
	/////////////////////////////
	
	/////////////////////////////
	// SETTERS
	/////////////////////////////
	
	
	/////////////////////
	// DATA MANAGEMENT
	////////////////////	
	
	protected function _run_before_create(){ 
		if(!isset($this->disclosed)) $this->disclosed = time();
		return true; 
	}	
	
	protected function _values_for_save(){
		$values_for_save = parent::_values_for_save();
		
		//if the message has changed, set the data from the mailbox
		if(array_key_exists(static::related_foreign_key('message'), $this->altered_values)){
			$mailbox = $this->message->mailbox;
			$values_for_save['username'] = $mailbox->name;
			$values_for_save['facility'] = $mailbox->facility('name');
		}

		return $values_for_save;
	}
	

	/**
	* Validates that the values match an attachment that requires disclosure on an existing message; if no recipient is specified, the message must be a draft.
	* @return boolean
	*/
	protected function _values_are_valid_for_create_and_update(){	
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		
		$message_foreign_key = static::related_foreign_key('message');
		
		$required_fields = array( $message_foreign_key, 'purpose', 'ssn', 'disclosed');
		foreach($required_fields as $required_field){
			if($this->property_is_empty($required_field)) return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving'); 
		}
		
		//make sure that the message exists
		$message = Message::find_one($this->$message_foreign_key);
		if(!Message::is_an_entity($message)) return $this->error->warning('There is no message with id '.$this->$message_foreign_key);
		if($this->property_is_empty('recipient') && !$message->draft) return $this->error->warning($message->describe().' is not a draft');
		
		//verify that there isn't already a disclosure for this attachment or patient
		//todo - could separate this into an overloadable hook so taht attachment disclosure can override it
		if(isset($this->id)) static::db()->where(static::$primary_key.' !='. $this->id);
		if(!$this->property_is_empty('hash')){
			if(static::exists(array($message_foreign_key => $this->$message_foreign_key, 'hash' => $this->hash, 'recipient' => $this->recipient))){
				return $this->error->warning('There is already a disclosure for message#'.$this->$message_foreign_key.' with hash='.$this->hash.' and recipient='.$this->error->describe($this->recipient));		
			}
		}else{
			Disclosure::db()->where('hash IS NULL');
			if(static::exists(array($message_foreign_key => $this->$message_foreign_key, 'ssn' => $this->ssn, 'recipient' => $this->recipient)))
				return $this->error->warning('There is already a disclosure for message#'.$this->$message_foreign_key.' with ssn='.$this->ssn.' and recipient='.$this->error->describe($this->recipient));				
		}
		
		//make sure that given/family names are provided if this isn't a disclosure for a parseable attachmetn
		if(!$this->property_exists('attachment') || !is_a($this->attachment, 'Patient_document_attachment')){
			foreach(array('first', 'last') as $required_field){
				if($this->property_is_empty($required_field)) return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving'); 
	
			}
		}		

		return true;
	}
	
///////////////////////////
// STATIC METHODS
///////////////////////////	
		
	public static function update_or_create_for_message($message, $values){
		if(!Message::is_an_entity($message)) return should_be('message entity', $message);
		if(!is_array($values) || empty($values)) return should_be('nonempty array', $values);
		
		$message_foreign_key = static::related_foreign_key('message');
		$values[$message_foreign_key] = $message->id;
		
		if(!empty($values['hash'])){
			$disclosure = Attachment_disclosure::find_one(array_intersect_key($values, array_flip(array($message_foreign_key, 'hash', 'recipient'))));
		
			if(Attachment_disclosure::is_an_entity($disclosure)){
				return Attachment_disclosure::update($disclosure->id, $values);
			}
			return Attachment_disclosure::create($values);
		}
		
		Disclosure::db()->where('hash IS NULL'); //make sure the check that this is unique includes the hash check	
		$disclosure = Disclosure::find_one(array_intersect_key($values, array_flip(array($message_foreign_key, 'ssn', 'recipient'))));
		
		if(Disclosure::is_an_entity($disclosure)){
			return Disclosure::update($disclosure->id, $values);
		}
		
		Disclosure::db()->where('hash IS NULL'); //make sure the check that this is unique includes the hash check	
		return Disclosure::create($values);
	}	
		
}
